/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xladata.c
 *       tracememory and trigger functions
 * -----------------------------------------------------------------*/

#include <xtypedef.h>
#include <xpciapi.h>
#include <xcmd.h>
#include <xlasterr.h>

#include <xdynamic.h>
#include <xerrcapi.h>
#include <xiocommo.h>
#include <xladata.h>
#include <xregcons.h>
#include <xsession.h>
#include <xversion.h>
#include <xaddrmap.h>

#include <xio.h>
/* 
  chunk size for saving tracememory contents to file
  Caution: If you increase this, you may have to increase
  MAX_XFER_SIZE in ezusbsys.c too !
*/
#define TRACE_DUMP_CHUNK    4096UL     /* Mephisto based */
#define TRACE_DUMP_CHUNK_30 0x400000UL /* trace lines for Faust cards */

/*****************************************************************************
 * declaration of static functions
 *****************************************************************************/

static bx_errtype TraceTrigInfoRead (bx_handletype handle,
                     bx_triginfo_type * pTrigInfo);

/*---------------------------------------------------------------------------*
 * BestXTraceDefaultWrite
 *
 * Purpose: Writes all trace properties with default values to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTraceDefaultWrite(
    bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXTraceDefaultWrite [tracedefwrite]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;
  const bx_param_infotype *ParamInfo;
  const bx_generic_infotype *GenInfo;

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);

    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRACE_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_TRACE_GEN, i, &ParamInfo,
          (bx_int32) BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXTraceWrite(handle, ParamInfo->proptyp.traceprop,
          ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTraceWrite
 *
 * Purpose: sets the properties for the tracememory counter
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTraceWrite(
  bx_handletype handle,
  bx_tracetype  prop,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME("BestXTraceWrite [tracewrite]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);
    
    /* range check */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_TRACE_GEN,(bx_int32)prop,val));
    
    switch (prop)
    {
      case BX_TRACE_HEARTBEATMODE:
        break;
      case BX_TRACE_HEARTBEATVALUE:
        break;
      case BX_TRACE_TRIG_HISTORY:
        /* SW is zero based, i.e property is number of lines stored _after_ the 
           triggerpoint. HW needs at least a '1' (denotes triggerpoint).
        */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRC_TRIG_PL_REG,sizeof(bx_int32),val+1));
        break;
      default:
        break;
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTraceRead
 *
 * Purpose: gets the properties for the tracememory counter
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTraceRead(
    bx_handletype handle,
    bx_tracetype  prop,
    bx_int32 *val
)
{
  BX_DECLARE_FUNCNAME("BestXTraceRead [traceread]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);

    BX_TRY(BestXParamProbe(handle, BX_PARAM_TRACE_GEN, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    switch (prop)
    {
      case BX_TRACE_HEARTBEATMODE:
        break;
      case BX_TRACE_HEARTBEATVALUE:
        break;
      case BX_TRACE_TRIG_HISTORY:
        /* SW is zero based, i.e property is number of lines stored _after_ the 
           triggerpoint. HW needs at least a '1' (denotes triggerpoint).
        */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_TRC_TRIG_PL_REG,sizeof(bx_int32),val));
        (*val)--;
        break;
      default:
        break;
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTraceRun
 *
 * Purpose: runs the analyzer/trace
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXTraceRun(
    bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXTraceRun [tracerun]");
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 value;        /* gen. purpose value */

    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);

    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_ANALYZER));

    /* reset trace memory registers */
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_TR_LOCAL_REG_RSTN, 4UL, 1UL));
    /* reset trace memory state machine */
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_TR_LOCAL_SM_RSTN, 4UL, 1UL));

    if (BestXHasFaust(handle))
    {
      /* no piggyback board needed */
    }
    else
    {
      BX_TRY(BestXStatusRead(handle, BX_STAT_PIGGYBACKID, &value));
      if (value != BX_PIGGY_ID_DT)
      {
        BX_E_ERROR_MSG_SET("BestXTraceRun: Cannot run without deep trace (option 100) board ");
        BX_TRY_FAIL(BX_E_ERROR);
      }
    }

    /* set DT address counter to 0*/

    /* reset FIFO, start precharge of SDRAM, write operation, init for PU */
    /* function enable */
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M), 2UL, 0x0023UL));

    /* set RAM address to 0 */
    if (BestXHasFaust(handle))
    {
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_DT_ADDR_HI_F, 2UL, 0UL));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_DT_ADDR_LO_F, 2UL, 0UL));
    }
    else
    {
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_DT_ADDR_HI_M, 4UL, 0UL));
    }

    /* start writing into SDRAM, general ok to write, release to write */
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M), 2UL, 2UL));
    /* not data is stored into the DT memory board */

    /* start trigger sequencer */
    BX_TRY(BestXSyncRegWrite(handle,

                 BX_REG_AN_EN_STATUS,
                 sizeof(bx_int16),
                 0x1,
                 0x1,

                 BX_REG_AN_EN_REG,
                 sizeof(bx_int16),
                 0x2,
                 0x2,

                 BX_REG_AN_EN_SREG,

                 BX_REG_AN_EN_FORCE_REG,
                 sizeof(bx_int16),
                 0 /* no soft force */));
    
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER));
    
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTraceStop
 *
 * Purpose: runs the analyzer/trace
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXTraceStop(
    bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXTraceStop [tracestop]");
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);

    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_ANALYZER));
    
    BX_TRY(BestXSyncRegWrite(handle,
                          
                BX_REG_AN_EN_STATUS,
                sizeof(bx_int16),
                0x1,
                0x1,
                          
                BX_REG_AN_EN_REG,
                sizeof(bx_int16),
                0x2,
                0x0,
                          
                BX_REG_AN_EN_SREG,

                BX_REG_AN_EN_FORCE_REG,
                sizeof(bx_int16),
                0 /* no soft force */ 
    ));
  
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER));  
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTraceDataRead
 *
 * Purpose: Gets the data and fills them into the array of longs
 *          passed down. Caller is responsible for allocating enough memory
 * NOTE   : This has been made a wrapper for BestXTraceDataReadAdv!
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTraceDataRead(
    bx_handletype handle,
    bx_int32 firstLine, 
    bx_int32 numLines,
    bx_int32 * data
)
{
  BX_DECLARE_FUNCNAME("BestXTraceDataRead [tracedataread]");

  return BestXTraceDataReadAdv (handle, firstLine, numLines, data, BX_TRUE, NULL);
}

/*---------------------------------------------------------------------------*
 * BestXTraceBytePerLineGet
 *
 * Purpose: returns the number of bytes per tracememory line            
 *          The information is taken out of the dynamic capabilities 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTraceBytePerLineGet(
    bx_handletype handle,
    bx_int32 * bytes_per_line
)
{
  /* holds the generic infotype for tracememory */
  const bx_generic_infotype *geninfo = NULL;

  BX_DECLARE_FUNCNAME("BestXTraceBytePerLineGet [tracebyteperlineget]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FCT_PARAM_NULL_POINTER(bytes_per_line);
    *bytes_per_line = 0;
    
    BX_LICENSECHECK(BX_CAPABILITY_CAPI);
    
    BX_TRY(BestXGenInfoGet(handle,BX_PARAM_TRACE_MEM, &geninfo));
    *bytes_per_line = geninfo->num_pages;
  }

  BX_ERRETURN(BX_TRY_RET);
}


/* TCL/SWIG helper function */
bx_errtype EXPORT BestXTraceBitPositionGet(
    bx_handletype handle,
    bx_signaltype signal,
    bx_int32 * position
)
{
  bx_int32 length; /* dummy */
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY(BestXTraceBitPosGet(handle,signal,position,&length));    
  }

  BX_ERRETURN(BX_TRY_RET);
}

/* TCL/SWIG helper function */
bx_errtype EXPORT BestXTraceBitLengthGet(
    bx_handletype handle,
    bx_signaltype signal,
    bx_int32 * length
)
{
  bx_int32 position; /* dummy */
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY(BestXTraceBitPosGet(handle,signal,&position,length));
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXTraceBitPosGet
 *
 * Purpose: Returns the bitposition fo a signal and the width of the         
 *           signal in bits. The information is taken out of 
 *           the dynamic capabilities                   
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTraceBitPosGet(
    bx_handletype handle,
    bx_signaltype signal,
    bx_int32 * position,
    bx_int32 * length
)
{
  const bx_param_infotype *parinfo;
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI); 
    
    *position = 0;
    *length = 0;
    BX_TRY(BestXParamInfoGet(handle, 
                             BX_PARAM_TRACE_MEM, 
                             (bx_int32) signal, 
                             &parinfo,
                             (bx_int32) BX_ENUM_SEARCH));

    *position = parinfo->min_val; /* we found the signal and here is pos */
    *length = parinfo->max_val;   /* and here the width */

  } /* BX_TRY_BEGIN */

  BX_TRY_CATCH; /* else...*/
  { 
    BestXLastErrorAllParamsSet(handle,
                   (bx_int32) BX_PARAM_TRACE_MEM,
                   (bx_int32) signal,
                   0UL,
                   0UL,
                   0UL);
  }

  BX_ERRETURN(BX_TRY_RET);
}


/* ------------------------------------------------------------------------
 * This function calculates the gap information and fills the array given
 * from the calling function.
 * ------------------------------------------------------------------------ */

bx_errtype EXPORT BestXTraceDump(
    bx_handletype handle,
    bx_charptrtype filename
)
{
  BX_DECLARE_FUNCNAME("BestXTraceDump [tracedump]");
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI); 
    
    BX_FCT_PARAM_NULL_POINTER_CHK(filename);
    
    /* lock the resource analyzer ...*/
    BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_ANALYZER));

    BX_TRY(BestXTraceDumpAdv(handle,
                    filename, 
                    ~0UL, 
                    ~0UL, 
                    NULL, 
                    0,
                    BX_WFM_COMPRESSED_1));
     /* unlock the resource analyzer... */
    BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/* This is the enhanced version of the BestXTraceDump().
   You can specify the firstLine(offset) and numLines (number of lines) you want to dump.
   The function pointer cb will be called periodically with an value between 1 and 100
   to indicate its progress.
   If cb returns with a value of zero the dumping will continue.
   In any other case the filedump will be cancelled; the file will not be deleted.
   The modebit reflects wether the file will be saved in performance mode or
   not - in the NO case, do not do performance uploads.
   It is also possible to specify the compression method (saving disk space).
*/

bx_errtype EXPORT BestXTraceDumpAdv(
    bx_handletype handle,
    bx_charptrtype filename,
    bx_int32 pretrig,
    bx_int32 posttrig,
    int (*cb) (int),
    int perfbit,
    int storemode)
{
  BX_DECLARE_FUNCNAME("BestXTraceDumpAdv");
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    FILE *fp;
    bx_int32 lcnt, length=0;
    bx_int32ptr data;
    bx_int32 i, j;
    bx_int32 longs_per_line;
    bx_int32 bytes_per_line;
    bx_int32 status=0, triggerpoint=0, dstart, dend;
    bx_int32 bus_width=64;
    bx_int32 bus_speed=0;
    bx_bool do_init;     /* switches between tracedataread w or wo setup */
    int bound;
    char *info=NULL;
    bx_int32 resetcode=0;
    bx_int32 mode=0;

    BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);

    BX_TRY_FCT_PARAM_NULL_POINTER(filename);

    /* read the number of bytes per line  */
    BX_TRY(BestXTraceBytePerLineGet(handle, &bytes_per_line));

    longs_per_line = bytes_per_line / sizeof(unsigned long);

    BX_TRY(BestXStatusRead(handle, BX_STAT_TRC_TRIGGER, &status));
    BX_TRY(BestXStatusRead(handle, BX_STAT_TRC_TRIGPOS, &triggerpoint));
    BX_TRY(BestXStatusRead(handle, BX_STAT_TRC_LINES, &length));
    BX_TRY(BestXStatusRead(handle, BX_STAT_BUSWIDTH, &bus_width));
    BX_TRY(BestXStatusRead(handle, BX_STAT_BUSSPEED, &bus_speed));
    BX_TRY(BestXStatusRead(handle, BX_STAT_RESETCODE, &resetcode));
    BX_TRY(BestXVersionRead(handle,BX_VERSION_CAPI,&info));

    //status=1;
    //triggerpoint=0;
    //length=0x200000;
    //length=0x4;  // 64 bytes
    //length=0x40; // 1024 bytes=16*64 bytes
    //length=0x1000;


    /* Check the values of pretrig and posttrig */
    if (!status)
    {
      posttrig = 0UL;
      triggerpoint = length;
    }
    else
    {
      if (((triggerpoint + posttrig) >= length) || (posttrig == ~0UL))
      {
        posttrig = length - triggerpoint - 1;
      }
    }

    if ((triggerpoint < pretrig) || (pretrig == ~0UL))
    {
      pretrig = triggerpoint;
    }

    /* calculate start and end points of dump */
    dstart = triggerpoint - pretrig;
    dend = triggerpoint + posttrig;

    if ((fp = BESTX_FOPEN(filename, "wb")) == (FILE *) 0)
    {
      BX_TRY_FAIL(BX_E_FILE_OPEN);
    }

    rewind(fp);

    data = (bx_int32 *) BestXMemMalloc((size_t) ((BestXHasFaust(handle)?TRACE_DUMP_CHUNK_30:TRACE_DUMP_CHUNK) * longs_per_line * 4));

    if (NULL == data)
    {
      BX_TRY_FAIL(BX_E_HOST_MEM_FULL);
    }

    /* Do some simplification for packing properties into mode */
    bus_speed /= (bx_int32)1E6; // convert into MHz 
    bus_width = (bus_width == 64UL ? 1UL : 0);  /* 1=64bit */

    // mode=(perfbit[0],buswidth[1],triggered[2],busspeed[3-10],resetcode[11-14],reserved[15-31])
    mode = perfbit<<0 | bus_width<<1 | status<<2 | bus_speed<<3 | resetcode<<11;

    BESTX_FPRINTF(fp, "Trigger: 0x%lx Lines: 0x%lx Hardware: %s Mode: %lu SW: %s %s\n",
      pretrig,
      (length ? pretrig + posttrig + 1 : 0),
      bx_handlearray[handle].hwinfo.product_string,
      mode,
      info,
      (storemode == BX_WFM_COMPRESSED_1 ? BX_WFM_COMP_1_STRING : ""));

    do_init = BX_TRUE;

    if (length)
    {
      bx_int32 saveAddr;

      lcnt = dstart;

      while (lcnt <= dend)
      {
        bound = (int) min ((dend - lcnt + 1),(BestXHasFaust(handle)?TRACE_DUMP_CHUNK_30:TRACE_DUMP_CHUNK));
        BX_TRY(BestXTraceDataReadAdv(handle, lcnt,(bx_int32) bound,data, do_init, &saveAddr));
        do_init = BX_FALSE;

        if (BX_TRY_RET != BX_E_OK)
        {
          BestXFree(data);
          BESTX_FCLOSE(fp);
          BX_ERRETURN(BX_TRY_RET);
        }

        if(storemode == BX_WFM_COMPRESSED_1)
        {
          fwrite((void *)data,1,bound * longs_per_line * 4,fp);
        }
        else /* standard is still BX_WFM_STANDARD which means no compression */
        {
          for (i = 0; i < (bx_int32) bound; i++)
          {
            for (j = 0; j < longs_per_line; j++)
            {
              BESTX_FPRINTF(fp, "%08lx ", data[(size_t) (i * longs_per_line + j)]);
            }
            BESTX_FPRINTF(fp, "\n");
          }
        }

        lcnt += bound;

        if (cb)
        {
          if ((*cb) ((int) (100 * (lcnt - dstart) / (dend - dstart +1))))
          {
            lcnt = ~0UL;
          }
        }
      }
    }

    free(data);
    fclose(fp);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*****************************************************************************
 * DLL-wide functions
 *****************************************************************************/
/*---------------------------------------------------------------------------*
 * bx_errtype BestXTraceStatusRead (bx_handletype handle,
 *
 * Purpose    : read trace mem specific info
 *---------------------------------------------------------------------------*/
bx_errtype BestXTraceStatusRead (bx_handletype handle,
                 bx_statustype prop,
                 bx_int32ptr pValue)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN {
    bx_triginfo_type trigInfo;
    BX_TRY(TraceTrigInfoRead(handle, &trigInfo));

    switch (prop)
    {
     case BX_STAT_TRC_MEMFULL:
      *pValue = trigInfo.m_memFull;
      break;
     case BX_STAT_TRC_TRIGPOS:
      *pValue = trigInfo.m_triggerPoint;
      break;
     case BX_STAT_TRC_LINES:
      *pValue = trigInfo.m_linesCaptured;
      break;
     default:
      BX_TRY_ERROR(BX_E_INVALID_CASE);
      break;
    }
  }

  return (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXTraceMemorySize (bx_handletype handle,
 *
 * Purpose    : retrieve number of lines in tracemem
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXTraceMemorySize (bx_handletype handle,
                    bx_int32ptr pSize)
{
  BX_TRY_VARS_NO_PROG;
  
  BX_TRY_BEGIN 
  {
    bx_int32 value;

    assert (pSize);
    
    if (BestXHasFaust(handle))
    {
      *pSize = 4 << (20UL); /* 4 Meg samples */
    }
    else
    {
      BX_TRY (BestXStatusRead (handle, BX_STAT_PIGGYBACKID, &value));

      if (value == 1UL)        /* #100 board found! */
      {
        /* read DT version register */
        BX_TRY(BestXDirectRegRead(handle, BX_REG_DT_VERSION_M, 2UL, &value));
        /* lower byte is version, higher byte is size */
        switch (value & 0x00ffUL)
        {
          case 0x00:        /* preliminary version: size is 2Meg fix */
            *pSize = 2UL << 20UL;
            break;
          default:
            *pSize = (value & 0xFF00UL) << (12UL);/* 20 (== meg) minus 8 */
        }
      }
      else /* we do not have a trace memory on board! */
      {
        *pSize = 0;
      }
    }
  }
  
  return (BX_TRY_RET);
}

/*****************************************************************************
 * static function definitions
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * bx_errtype BestXTraceDataReadAdv(
 *
 * Purpose    : this function does the actual work:
 *                depending on doInit, it initializes the deep trace completely
 *                or does a "quick" access
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXTraceDataReadAdv(
  bx_handletype handle,
  bx_int32 firstLine,
  bx_int32 numLines,
  bx_int32 * data,
  bx_bool doInit,
  bx_int32ptr pTraceAddr
  )
{
  BX_DECLARE_FUNCNAME("BestXTraceDataReadAdv");
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_triginfo_type trigInfo;
    bx_int32 value;        /* general purpose value */
    bx_int32 traceAddr, traceAddrLo, traceAddrHi; /* absolute tracemem addr */
    bx_int32ptr dataPtr = data;
    bx_int32 bytes_per_line=0;
    
    if (doInit)
    {
      BX_LICENSECHECK(BX_CAPABILITY_ANALYZER|BX_CAPABILITY_CAPI);
      
      BX_TRY_FCT_PARAM_NULL_POINTER (data);

      /* lock the resource analyzer ...*/
      BX_TRY_PROGRESS(BestXResourceLock(handle, BX_RESLOCK_ANALYZER));

      /* stop trigger sequencer */
      BX_TRY (BestXTraceStop(handle)); /* stop trace memory */

      /* get info about trigger position etc. */
      BX_TRY (TraceTrigInfoRead(handle, &trigInfo));

      //trigInfo.m_linesCaptured=0x200000;
      //trigInfo.m_linesCaptured=0x8;
      //trigInfo.m_linesCaptured=0x40; // 1024 bytes=16*64 bytes
      //trigInfo.m_linesCaptured=0x1000; // 1024 bytes=16*64 bytes
      //trigInfo.m_memFull=1;
      //trigInfo.m_traceOffset=0;
      //trigInfo.m_triggerPoint=0;


      /* check parameters */
      BX_TRY_FCT_PARAM (firstLine, firstLine >= trigInfo.m_linesCaptured );
      BX_TRY_FCT_PARAM_RANGE (numLines, 1UL, trigInfo.m_linesCaptured );

      /* handle DT board: */
       
      /* read status of DT FIFO */
      BX_TRY (BestXDirectRegRead(handle, (BestXHasFaust(handle)?BX_REG_DT_STATUS_F:BX_REG_DT_STATUS_M), 2UL, &value));
      if ( (value & 0x001UL) == 0UL)
      {
        /* We should never get here */
        BX_TRY_FAIL (BX_E_TBD);    /* FIFO not empty yet... */
      }

      /* set SDRAM address */
      traceAddr = (trigInfo.m_traceOffset + firstLine);
    }
    else
    {
      BX_TRY_FAIL (pTraceAddr != NULL ? BX_E_OK : BX_E_TBD);
      /* we do the 'quick access' type */
      traceAddr = *pTraceAddr;
    }
    
    /* This access (precharge, i.e. set bit0) needs to be executed only once after powerup.
       Usually done by firmware once during powerup. 
       BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),2UL, 0x21));
    */
    
    /* This access makes sure, that the following
       write accesses to address, min and max registers will work correct. */
    BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),2UL, 0x0));

    /* Set startaddress (offset into tracemem) */
    traceAddrLo = traceAddr &  0x0000FFFFUL;
    traceAddrHi = (traceAddr & 0x003F0000UL) >> 16UL;
    BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M), 2UL, traceAddrLo));
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M), 2UL, traceAddrHi));

    /* Get number of bytes per traceline from dyncaps. This is Mode (and HW) dependent !!
       Mode1: 0x10, Mode2(DDR): 0x18, Mode2(QDR): ??? */ 
    BX_TRY(BestXTraceBytePerLineGet(handle,&bytes_per_line));

    if (doInit)
    {
      /* set data width (read full 128/192 bit) */
      BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_START_F:BX_REG_DT_LATCH_READ_START_M), 2UL,0UL));
      BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_STOP_F:BX_REG_DT_LATCH_READ_STOP_M), 2UL, bytes_per_line/2-1));
      
      /* reset FIFO */
      BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M), 2UL, 0x0020));
    }
    
    /* set read mode */
    if (BestXHasFaust(handle) && bx_handlearray[handle].port==BX_PORT_USB)
    {
      /* Bit 6 is needed for special Fast mode */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_DT_CONTROL_F,2UL,0x0026|1<<6));
    }
    else
    {
      BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M), 2UL, 0x0026));
    }

    /* now read the tracememory */
    if (BestXHasMephisto(handle))
    {
      BX_TRY (BestXDirectRegBlockRead(handle,
                    BX_REG_DT_DATAREAD_HI_M,      /* address */
                    4UL,                          /* regsize */
                    1UL,                          /* auto increment */
                    (bx_int8ptr) data,
                    4UL,                          /* data size */
                    numLines * bytes_per_line));  /* num of bytes */
    }
    else
    {
      BX_TRY (BestXDirectRegBlockRead(handle,
                    BX_REG_DT_DATAREAD_F,         /* address */
                    2UL,                          /* regsize */
                    1UL,                          /* auto increment */
                    (bx_int8ptr) data,
                    4UL,                          /* data size */
                    numLines * bytes_per_line));  /* num of bytes */
    }
  
    if (doInit)
    {
      /* unlock the resource analyzer... */
      BX_TRY_PROGRESS(BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER));
    }

    if (pTraceAddr)
    {
      *pTraceAddr = traceAddr+numLines; /* save it for next, quick, access */
    }
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Resource locking OK */
      BX_TRY_PASSED
      {
        /* Resource unlocking OK */
      }
      BX_TRY_FAILED
      {
        /* some function returned an error */
        (void)BestXResourceUnlock(handle, BX_RESLOCK_ANALYZER);
      }
    }
    BX_TRY_FAILED
    {
      /* Resource locking failed */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype TraceTrigInfoRead (bx_handletype handle,
 *
 * Purpose    : 
 *---------------------------------------------------------------------------*/
static bx_errtype TraceTrigInfoRead (bx_handletype handle,
                     bx_triginfo_type * pTrigInfo)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN {
    bx_int32 traceMemSize, wordCount, addrCount, trigPos, trigPrel;
    
    BX_TRY (BestXTraceMemorySize(handle, &traceMemSize));
    BX_TRY_FAIL (traceMemSize > 0UL ? BX_E_OK : BX_E_INTERNAL_RETURN);


#if 0 /* chris (see xboard.c, hwclass.cxx, trigger_info() in GUI) */
    /* check if tracemem has stopped */
    BX_TRY (BestXDirectRegRead(handle, BX_REG_TR_STATUS_REG,    4UL, &value));
    if ((value & 3UL) != 3UL)
    {
      /* In order to call this function, 
         trace needs to be triggered and stopped  */
      BX_TRY_FAIL(BX_E_INTERNAL_RETURN);
    }
#endif

    /* read number of words in tracemem */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_M_W_CNTR_REG,  4UL, &wordCount));

    /* read value of deep trace address counter */
    if (BestXHasFaust(handle))
    {
      bx_int32 addrCounthi=0;
      bx_int32 addrCountlo=0;
      BX_TRY(BestXDirectRegRead(handle,BX_REG_DT_LAST_W_ADDR_HI_F,  2UL, &addrCounthi));
      BX_TRY(BestXDirectRegRead(handle,BX_REG_DT_LAST_W_ADDR_LO_F,  2UL, &addrCountlo));
      addrCount= (addrCounthi<<16 & 0xffff0000) | (addrCountlo & 0x0000ffff);
    }
    else
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_DT_LAST_W_ADDR_HI_M,  4UL, &addrCount));
    }

    /* We need to increment addrCount by 1 because the value copied to 
    BX_REG_DT_W_ADDR when the trace stops is actually the current address counter -1 */
    addrCount++;

    /* read trigger pos */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_TRC_TRIG_CNTR_REG, 4UL,&trigPos));
    /* read trigger preload value */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_TRC_TRIG_PL_REG, 4UL, &trigPrel));
    
    /* compute number of lines captured and address offset*/
    if ( wordCount < traceMemSize )
    {
      pTrigInfo->m_memFull = 0;
      pTrigInfo->m_linesCaptured = wordCount;
      pTrigInfo->m_traceOffset = 0UL;
    }
    else
    {
      pTrigInfo->m_memFull = 1;
      pTrigInfo->m_linesCaptured = traceMemSize;
      pTrigInfo->m_traceOffset = addrCount;
    }

    /* The trigger point is set to 0 if the trace memory is empty! */
    pTrigInfo->m_triggerPoint = wordCount ? (pTrigInfo->m_linesCaptured - (trigPrel - trigPos)) : 0;

  }

  BX_TRY_CATCH
  {
    /* error condition? set to reasonable values... */
    memset (pTrigInfo, 0UL, sizeof(bx_triginfo_type));
    if (BX_TRY_RET == BX_E_INTERNAL_RETURN)
    {
      BX_TRY_RET = BX_E_OK;
    }
  }

  return (BX_TRY_RET);
}


/* --------------------------------------------------------------------------
 * Fills trace memory with a caller-defined pattern.  The meaning of
 * params value1 and value2 are dependent on the pattern as follows;
 * pattern          value1              value2
 * ==============   =================== =========================
 * BX_PATT_CONST     mem set to this     ignored
 * BX_PATT_TOGGLE    1st var             2nd var
 * BX_PATT_COUNTER   starting val.       0==count down, else count up
 * BX_PATT_WALKBIT0  ignored             ignored
 * BX_PATT_WALKBIT1  ignored             ignored
 * BX_PATT_RANDOM    seed                ignored
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXTraceRawPatternFill(
    bx_handletype handle,
    bx_int32 startAddr,         /* 22 resp. 20 bit line address, usually 0 */
    bx_int32 numLines,        
    bx_int16 value1,            /* meaning depends on pattern */
    bx_int16 value2,            /* same */
    bx_int16 mode,             
    bx_dmempatttype pattern     /* type of pattern to use */
)
{
  BX_DECLARE_FUNCNAME("BestXTraceRawPatternFill [tracerawpatternfill]");

  BX_TRY_VARS;

  bx_int16ptr data=NULL;
  const bx_generic_infotype *GenericDataMemInfo;
  bx_int32 MaxNumWords;
  bx_int32 i;
  bx_int32 words_per_line; /* number of words to write per line */
  bx_int32 numWords;       /* number of words to write alltogether */

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    
    /* Get number of words to write */
    words_per_line=(BestXHasFaust(handle)?12:8); /* independent of mode ! */
    numWords=words_per_line*numLines;

    /* Get size of tracememory in words */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRACE_MEM, &GenericDataMemInfo));
    MaxNumWords = GenericDataMemInfo->depth*words_per_line;

    if (startAddr*words_per_line+numWords>MaxNumWords)
    {
      BX_E_ERROR_MSG_SET("BestXTraceRawPatternFill: Parameters exceed tracememory size")
      BX_TRY_FAIL(BX_E_ERROR);
    }

    /* Allocate word array */
    data=(bx_int16ptr) BestXMemCalloc(numWords,2);
    BX_TRY_FAIL(data?BX_E_OK:BX_E_HOST_MEM_FULL);

    switch (pattern)
    {
      case BX_PATT_CONST32:
        for (i=0;i<numWords;i++) data[i]=value1; 
        break;
      case BX_PATT_TOGGLE32:
        for (i=0;i<numWords;i+=2)
        {
          data[i]=value1; 
          data[i+1]=value2; 
        }
        break;
      case BX_PATT_COUNTER32:
        if (value2)
        {
          /* going up */         
          for (i=0;i<numWords;i++) data[i]=(bx_int16)(value1+i);
        }
        else
        {
          /* going down */
          for (i=0;i<numWords;i++) data[i]=(bx_int16)(value1-i);
        }
        break;
      case BX_PATT_WALKBIT032:
        for (i=0;i<numWords;i++)
        {
          data[i]= ~((bx_int16)(1<<(i%16)));
        }
        break;
      case BX_PATT_WALKBIT132:
        for (i=0;i<numWords;i++)
        {
          data[i]= (bx_int16)(1<<(i%16));
        }
        break;
      case BX_PATT_RANDOM:
        srand(value1);
        for (i=0;i<numWords;i++)
        {
          /* Note: RAND_MAX is usually 0x7fff */
          data[i]= ((((bx_int16)rand())<<1) & 0xfffe) |   /* Bit 15-1  */
                   ((((bx_int16)rand())<< 0) & 0x1);      /* Bit  0    */
        }
        break;
      default:
        BX_E_ERROR_MSG_SET("BestXTraceRawPatternFill: Invalid value for parameter pattern")
        BX_TRY_FAIL(BX_E_ERROR);
        break;
    }

#if 1  /* Write reference file to compare against file written by BestXTraceRawDump() */
    {
      FILE *fp=NULL;     
      bx_int32 byteswritten;

      /* Open file for binary write */
      if ((fp = BESTX_FOPEN("c:\\tracerawreference.bin", "wb")) == (FILE *) NULL)
      {
        BX_TRY_FAIL(BX_E_FILE_OPEN);
      }

      if (BestXIsHostLittleEndian())
      {
        /* Convert word- to byte array */
        for (i=0;i<numWords;i++)
        {
          /* Swap bytes (if host is little endian) */
          data[i] = (data[i]>>8 & 0x00ff) | (data[i]<<8 & 0xff00);
        }
      }

      byteswritten=fwrite((void *)data,1,numWords*2,fp);
      BESTX_FCLOSE(fp);fp=NULL;

      BX_E_ERROR_MSG_SET("BestXTraceRawDump: Error while writing to ref file")
      BX_TRY_FAIL(byteswritten==numWords*2?BX_E_OK:BX_E_ERROR);

      if (BestXIsHostLittleEndian())
      {
        /* Convert word- to byte array */
        for (i=0;i<numWords;i++)
        {
          /* Swap bytes (if host is little endian) */
          data[i] = (data[i]>>8 & 0x00ff) | (data[i]<<8 & 0xff00);
        }
      }
    }
#endif    

    /* Now program trace memory */
    BX_TRY(BestXTraceRawWrite(handle,startAddr,data,numWords,mode));

    BestXMemFree((void **)&data);
  }
  BX_TRY_CATCH
  {
    if (data) 
    {
      BestXMemFree((void **)&data);
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


bx_errtype EXPORT BestXTraceRawWrite(
    bx_handletype handle,
    bx_int32 startAddr,          /* 22 resp. 20 bit line address, usually 0 */
    bx_int16ptr data,   
    bx_int32 numWords,           /* size of data in words, must be multiple of words_per_line ! */
    bx_int32 mode
)
{
  BX_DECLARE_FUNCNAME("BestXTraceRawWrite [tracerawwrite]");

  BX_TRY_VARS;

  const bx_generic_infotype *GenericDataMemInfo;
  bx_int32 MaxNumWords;
  bx_int32 i,dw;
  bx_int32 traceAddrLo,traceAddrHi;
  bx_int32 words_per_line; 

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    
    words_per_line=(BestXHasFaust(handle)?12:8); /* independent of mode ! */

    BX_FCT_PARAM_ALIGNMENT_CHK(numWords,words_per_line);

    /* Get size of tracememory in words */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRACE_MEM, &GenericDataMemInfo));
    MaxNumWords = GenericDataMemInfo->depth * words_per_line;

    if (startAddr*words_per_line+numWords>MaxNumWords)
    {
      BX_E_ERROR_MSG_SET("BestXTraceRawWrite: Parameters exceed tracememory size")
      BX_TRY_FAIL(BX_E_ERROR);
    }

    /* Now program trace memory */
    
    /* This access (precharge, i.e. set bit0) needs to be executed only once after powerup.
       Usually done by firmware once during powerup. 
       BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),2UL, 0x21));
    */
    
    /* This access makes sure, that the following
       write accesses to address, min and max registers will work correct. */
    BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),2UL, 0x0));

    /* Set startaddress */
    traceAddrLo = startAddr &  0x0000FFFFUL;
    traceAddrHi = (startAddr & 0x003F0000UL) >> 16UL;
    BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M), 2UL, traceAddrLo));
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M), 2UL, traceAddrHi));

    /* Set data width (read full 128/192 bit), independent of mode */
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_START_F:BX_REG_DT_LATCH_READ_START_M), 2UL,0UL));
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_STOP_F:BX_REG_DT_LATCH_READ_STOP_M), 2UL, words_per_line-1));
    
    /* Set write mode */
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M), 2UL, 0xa));

    /* Now write the tracememory */
    if (BestXHasMephisto(handle))
    {
      if (mode==0)
      {
        for (i=0;i<numWords;i+=2)
        {
          dw=(((bx_int32)data[i])<<16)|((bx_int32)data[i+1]);
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_DT_DATAREAD_HI_M,4,dw));
        }
      }
      else
      {
        BX_TRY (BestXDirectRegBlockWrite(handle,
                    BX_REG_DT_DATAREAD_HI_M,      /* address */
                    4UL,                          /* regsize */
                    1UL,                          /* auto increment */
                    (bx_int8ptr) data,
                    2UL,                          /* data size */
                    numWords*2));                 /* num of bytes */
      }
    }
    else
    {
      if (mode==0)
      {
        for (i=0;i<numWords;i++)
        {
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_DT_DATAREAD_F,2,data[i]));
        }
      }
      else
      {
        if (!BestXIsMode2(handle))
        {
          /* Faust based FW has a special fast mode implemented.
             The setup of this mode needs to know the width of 
             the tracememory line, which is determined by the current mode.
             So when we are in mode1, the setup programs 
             the tracewidth correspondingly thus ignoring the fact that
             this functions wants to write the whole tracelines, i.e. 
             is mode independent. This leads to wrong datawrite.
           */
          BX_E_ERROR_MSG_SET("BestXTraceRawWrite: Cannot write in SDR mode. Set mode parameter to zero (slow!) or make sure card is in DDR/QDR mode")
          BX_TRY_FAIL(BX_E_ERROR);
        }

        BX_TRY (BestXDirectRegBlockWrite(handle,
                    BX_REG_DT_DATAREAD_F,         /* address */
                    2UL,                          /* regsize */
                    1UL,                          /* auto increment */
                    (bx_int8ptr) data,
                    2UL,                          /* data size */
                    numWords*2));                 /* num of bytes */
      }
    }
  } /* TRY */

  BX_ERRETURN(BX_TRY_RET);
}

bx_errtype EXPORT BestXTraceRawRead(
    bx_handletype handle,
    bx_int32 startAddr,          /* 22 resp. 20 bit line address, usually 0 */
    bx_int16ptr data,   
    bx_int32 numWords,        /* size of data in words, must be multiple of words_per_line ! */
    bx_int32 mode
)
{
  BX_DECLARE_FUNCNAME("BestXTraceRawRead [tracerawread]");

  BX_TRY_VARS;

  const bx_generic_infotype *GenericDataMemInfo;
  bx_int32 MaxNumWords;
  bx_int32 i,dw;
  bx_int32 traceAddrLo,traceAddrHi,words_per_line;
  
  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    
    words_per_line=(BestXHasFaust(handle)?12:8); /* independent of mode ! */

    BX_FCT_PARAM_ALIGNMENT_CHK(numWords,words_per_line);

    /* Get size of tracememory in words */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRACE_MEM, &GenericDataMemInfo));
    MaxNumWords = GenericDataMemInfo->depth * words_per_line;

    if (startAddr*words_per_line+numWords>MaxNumWords)
    {
      BX_E_ERROR_MSG_SET("BestXTraceRawRead: Parameters exceed tracememory size")
      BX_TRY_FAIL(BX_E_ERROR);
    }

    /* Now read trace memory */
    
    /* This access (precharge, i.e. set bit0) needs to be executed only once after powerup.
       Usually done by firmware once during powerup. 
       BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),2UL, 0x21));
    */
    
    /* This access makes sure, that the following
       write accesses to address, min and max registers will work correct. */
    BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),2UL, 0x0));

    /* Set startaddress */
    traceAddrLo = startAddr &  0x0000FFFFUL;
    traceAddrHi = (startAddr & 0x003F0000UL) >> 16UL;
    BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M), 2UL, traceAddrLo));
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M), 2UL, traceAddrHi));

    /* Set data width (read full 128/192 bit), independent of mode */
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_START_F:BX_REG_DT_LATCH_READ_START_M), 2UL,0UL));
    BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_STOP_F:BX_REG_DT_LATCH_READ_STOP_M), 2UL, words_per_line-1));

    /* Set read mode */
    BX_TRY(BestXDirectRegWrite(handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),2UL, 0x20));
    if (BestXHasFaust(handle) && bx_handlearray[handle].port==BX_PORT_USB)
    {
      /* Bit 6 is needed for special Fast mode */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_DT_CONTROL_F,2UL,0x0026|1<<6));
    }
    else
    {
      BX_TRY(BestXDirectRegWrite(handle, (BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M), 2UL, 0x0026));
    }

    /* Now read the tracememory */
    if (BestXHasMephisto(handle))
    {
      if (mode==0)
      {
        for (i=0;i<numWords;i+=2)
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_DT_DATAREAD_HI_M,4,&dw));
          data[i]=(bx_int16)(dw>>16 & 0xffff);
          data[i+1]=(bx_int16)(dw & 0xffff);
        }
      }
      else
      {
        BX_TRY (BestXDirectRegBlockRead(handle,
                    BX_REG_DT_DATAREAD_HI_M,      /* address */
                    4UL,                          /* regsize */
                    1UL,                          /* auto increment */
                    (bx_int8ptr) data,
                    2UL,                          /* data size */
                    numWords*2));                 /* num of bytes */
      }
    }
    else
    {
      if (mode==0)
      {
        /* chris TBD: 
           This case seems to fail with E2930A/B cards
           when using the USB port (SDR and DDR) ! */
        for (i=0;i<numWords;i++)
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_DT_DATAREAD_F,2,&dw));
          data[i]=(bx_int16)dw;
        }
      }
      else
      {
        if (!BestXIsMode2(handle))
        {
          /* Faust based FW has a special fast mode implemented.
             The setup of this mode needs to know the width of 
             the tracememory line, which is determined by the current mode.
             So when we are in mode1, the setup programs 
             the tracewidth correspondingly thus ignoring the fact that
             this functions wants to read the whole tracelines, i.e. 
             is mode independent. This leads to wrong dataread.
           */
          BX_E_ERROR_MSG_SET("BestXTraceRawRead: Cannot read in SDR mode. Set mode parameter to zero (slow!) or make sure card is in DDR/QDR mode")
          BX_TRY_FAIL(BX_E_ERROR);
        }

        BX_TRY (BestXDirectRegBlockRead(handle,
                    BX_REG_DT_DATAREAD_F,         /* address */
                    2UL,                          /* regsize */
                    1UL,                          /* auto increment */
                    (bx_int8ptr) data,
                    2UL,                          /* data size */
                    numWords*2));                 /* num of bytes */
      }
    }
  } /* TRY */

  BX_ERRETURN(BX_TRY_RET);
}

bx_errtype EXPORT BestXTraceRawDump(
    bx_handletype handle,
    bx_int32 startAddr,         /* 22 resp. 20 bit line address, usually 0 */
    bx_int32 numLines,        
    bx_int16 mode,
    bx_charptrtype filename

  )
{
  BX_DECLARE_FUNCNAME("BestXTraceRawDump [tracerawdump]");

  BX_TRY_VARS;

  const bx_generic_infotype *GenericDataMemInfo;
  bx_int32 MaxNumWords;
  bx_int16ptr data=NULL;
  FILE *fp=NULL;
  bx_int32 byteswritten=0,i,words_per_line;
  bx_int32 numWords;       /* number of words to read alltogether */

  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    
    /* Get number of words to write */
    words_per_line=(BestXHasFaust(handle)?12:8); /* independent of mode ! */
    numWords=words_per_line*numLines;

    /* Get size of tracememory in words */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_TRACE_MEM, &GenericDataMemInfo));
    MaxNumWords = GenericDataMemInfo->depth*words_per_line;

    if (startAddr*words_per_line+numWords>MaxNumWords)
    {
      BX_E_ERROR_MSG_SET("BestXTraceRawDump: Parameters exceed tracememory size")
      BX_TRY_FAIL(BX_E_ERROR);
    }

    /* Open file for binary write */
    if ((fp = BESTX_FOPEN(filename, "wb")) == (FILE *) NULL)
    {
      BX_TRY_FAIL(BX_E_FILE_OPEN);
    }

    /* Allocate word array */
    data=(bx_int16ptr) BestXMemMalloc(numWords*2);
    BX_TRY_FAIL(data?BX_E_OK:BX_E_HOST_MEM_FULL);

    /* Now read trace memory */
    BX_TRY(BestXTraceRawRead(handle,startAddr,data,numWords,mode));

    if (BestXIsHostLittleEndian())
    {
      /* Convert word- to byte array */
      for (i=0;i<numWords;i++)
      {
        /* Swap bytes (if host is little endian) */
        data[i] = (data[i]>>8 & 0x00ff) | (data[i]<<8 & 0xff00);
      }
    }

    /* Dump to file */
    byteswritten=fwrite((void *)data,1,numWords*2,fp);
    BESTX_FCLOSE(fp);fp=NULL;

    BX_E_ERROR_MSG_SET("BestXTraceRawDump: Error while writing to file")
    BX_TRY_FAIL(byteswritten==numWords*2?BX_E_OK:BX_E_ERROR);

    BestXMemFree((void **)&data);
  }
  BX_TRY_CATCH
  {
    if (data) 
    {
      BestXMemFree((void **)&data);
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
